# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Models used by debusine wait tasks."""

import datetime as dt
from enum import StrEnum
from typing import override

from debusine.tasks.models import (
    BaseDynamicTaskData,
    BaseTaskData,
    LookupSingle,
)


class DelayData(BaseTaskData):
    """In-memory task data for the Delay task."""

    delay_until: dt.datetime


class ExternalDebsignData(BaseTaskData):
    """In-memory task data for the ExternalDebsign task."""

    unsigned: LookupSingle


class ExternalDebsignDynamicData(BaseDynamicTaskData):
    """Dynamic data for the ExternalDebsign task."""

    unsigned_id: int

    @override
    def get_input_artifacts_ids(self) -> list[int]:
        return [self.unsigned_id]


class DenyAction(StrEnum):
    """The action to take when denying a Confirm task."""

    FAIL = "fail"
    ABORT = "abort"
    ABORT_WORKFLOW = "abort-workflow"


class ConfirmData(BaseTaskData):
    """In-memory task data for the Confirm task."""

    auto_confirm_if_no_failure: bool = False
    deny_action: DenyAction = DenyAction.ABORT_WORKFLOW
    confirm_label: str = "Continue the workflow"
    deny_label: str = "Abort the workflow"
